# -*- coding:utf-8 -*-
import os
from datetime import datetime

def collect_folder_and_file_names_tree(root_directory):
    """
    递归遍历指定目录及其所有子目录，收集所有文件夹名称和文件名，
    并将其格式化为目录树的字符串列表。
    使用字符线条（├──, └──, │）来表示层级关系。
    """
    tree_lines = []

    def _walk_and_format(current_path, prefix, is_last):
        """
        递归辅助函数，用于遍历并格式化目录树。
        :param current_path: 当前正在处理的路径
        :param prefix: 当前层级的前缀（用于绘制垂直线和缩进）
        :param is_last: 布尔值，表示当前项是否是其父目录下的最后一个子项
        """
        # 获取当前项的显示名称
        display_name = os.path.basename(current_path)
        if not display_name and current_path == root_directory: # 处理根目录本身（如 C:\ 或 /）
            display_name = current_path
        
        # 将当前项添加到树状视图列表
        if current_path == root_directory:
            tree_lines.append(display_name + os.sep) # 根目录直接显示名称，并加斜杠
        else:
            connector = "└── " if is_last else "├── "
            tree_lines.append(prefix + connector + display_name)

        # 如果当前项是目录，则继续处理其子项
        if os.path.isdir(current_path):
            # 为子项准备新的前缀
            child_prefix = prefix + ("    " if is_last else "│   ")

            try:
                # 获取当前目录下的所有项
                items = os.listdir(current_path)
            except PermissionError:
                tree_lines.append(child_prefix + "└── [Permission Denied]")
                return
            except OSError as e:
                tree_lines.append(child_prefix + f"└── [Error: {e}]")
                return

            # 分离目录和文件，并进行排序，确保输出顺序一致
            dirs = sorted([item for item in items if os.path.isdir(os.path.join(current_path, item))])
            files = sorted([item for item in items if os.path.isfile(os.path.join(current_path, item))])

            # 合并所有子项，先目录后文件
            all_children = dirs + files

            # 递归处理所有子项
            for i, item_name in enumerate(all_children):
                item_path = os.path.join(current_path, item_name)
                is_last_child = (i == len(all_children) - 1) # 判断是否是当前父目录的最后一个子项
                _walk_and_format(item_path, child_prefix, is_last_child)

    # 从根目录开始递归
    _walk_and_format(root_directory, "", True) # 根目录没有前缀，且在逻辑上是其自身上下文的最后一个

    return tree_lines

def save_to_txt_file(data_list, output_filename):
    """
    将列表数据写入到文本文件中，每行一个。
    """
    try:
        with open(output_filename, 'w', encoding='utf-8') as f:
            for item in data_list:
                f.write(item + '\n')
        print(f"✅ 目录树结构已成功保存到: {output_filename}")
    except Exception as e:
        print(f"❌ 写入文件失败: {e}")

def main():
    print("--- 文件夹和文件名收集工具 (目录树结构) ---")
    
    # 获取用户输入的目录路径
    root_folder = input("请输入要遍历的根目录路径（例如 D:\\MyProject 或 /Users/Me/Documents）：").strip('"').strip("'")

    # 检查输入的路径是否是一个有效的目录
    if not os.path.isdir(root_folder):
        print(f"❌ 错误: '{root_folder}' 不是一个有效的目录。请检查路径是否正确。")
        input("\n按回车键退出程序...")
        return

    print(f"\n开始收集 '{root_folder}' 中的目录树结构...")
    
    # 收集目录树结构
    tree_lines = collect_folder_and_file_names_tree(root_folder)
    
    if not tree_lines:
        print("ℹ️ 未在指定目录及其子目录中找到任何文件夹或文件。")
        input("\n按回车键退出程序...")
        return

    # 生成输出文件名，包含当前日期时间
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    output_file = f"collected_directory_tree_{timestamp}.txt"
    
    # 保存目录树结构到文件
    save_to_txt_file(tree_lines, output_file)
    
    print(f"\n共收集到 {len(tree_lines)} 行目录树信息。")
    print("------------------------")
    
    # 保持窗口打开，直到用户按下回车键
    input("\n按回车键退出程序...")

if __name__ == "__main__":
    main()

